<?php
namespace FormFields;

use SilverStripe\ORM\DataObject;
use SilverStripe\View\Requirements;
use SilverStripe\Forms\DropdownField;
use SilverStripe\Forms\TextField;
use SilverStripe\Forms\CheckboxField;
use SilverStripe\Core\Convert;
use FormFields\CustomisableFormPanel;
use SilverStripe\Forms\DateField;
use SilverStripe\Forms\NumericField;
use SilverStripe\Forms\OptionsetField;
use SilverStripe\Forms\TextareaField;
use SilverStripe\Forms\FileField;
use SilverStripe\Forms\SelectField;


class CustomisableFormField extends DataObject {

	/**
	 * Base class for customisable form field.
	 * Generic/inherited functions in here.
	 */

	private static $db = array(
		"Title"                 => "Varchar(255)",
		"Required"              => "Boolean",
		"FieldTitle"            => "Varchar(255)",
		"Sort"                  => "Int",
		"HideField"             => "Boolean",
		"Length"          		=> "Enum(array('Long','Medium','Short'))",
		"Label"                 => "Varchar(255)",
	);

	private static $field_labels = array(
		"Title" => "Name"
	);

	private static $table_name = 'CustomisableFormField';
	

	public function getFieldLengthClass(){
		switch($this->Length){
			case "Long":
				return "FieldWrapper";
			case "Medium":
				return "col span_2_of_3";
			case "Short":
				return "col span_1_of_3";
			default:
				return "FieldWrapper";
		}
	}
	private static $has_options = false;

	private static $has_upload = false;

	private static $remove_from_list = false;

	private static $base_field_type = "TextField";

	private static $has_one = array(
		"CustomisableFormPanel"  => CustomisableFormPanel::class,
		// "PrerequisiteField"      => CustomisableFormOptionField::class,
		// "PrerequisiteFieldValue" => CustomisableFormOptionFieldItem::class
	);

	private static $default_sort = "Sort";

	public function getCMSFields(){
		$fields = parent::getCMSFields();
		Requirements::CSS('BlockWidgetCustomisableForm/css/cms.css');
		$fields->removeByName("CustomisableFormPanelID");
		$fields->removeByName("FieldTitle");
		$fields->removeByName("Sort");

		$fields->addFieldToTab(
			"Root.Main",
			DropdownField::create("Length", "Field length", singleton(CustomisableFormField::class)->dbObject('Length')->enumValues())
				->setDescription("Please select the length of your field. If none is selected long field will return as default"));

		$fields->addFieldToTab(
			"Root.Main",
			TextField::create("Label", "Label")
				->setDescription("Optional label field. If left blank no label will display")
		);

		$fields->replaceField("Required", new CheckboxField("Required", "This field is required."));

		return $fields;
	}


	public function getLimitingFieldDataVar(){
		return "";
	}

	private static $summary_fields = array(
		"Title"             => "Title",
		"FieldTypeName"     => "Type"
	);

	public function parseValue($value){
		return $value;
	}

	public function getFieldTypeName(){
		return $this->singular_name();
	}

	public function getDataFieldTitle(){
		return $this->FieldTitle;
	}

	public function getCustomField(){
		$fieldType = $this->stat('base_field_type'); //gets SilverStripe\Forms\TextField
		$field = new $fieldType($this->getDataFieldTitle(), $this->Title, $value = ''); //creates SilverStripe\Forms\TextField Object ()
		$field->setAttribute("placeholder", $this->Title);
		if ($this->Required){
			$field->setAttribute("required", "required");
		}
		return $field;
	}

	public function Output(){
		return $this->renderWith($this->ClassName);
	}

	public function onBeforeWrite()
	{
		if ($this->isChanged("Title") || strlen($this->FieldTitle) == 0){
			$this->FieldTitle = convert::raw2url($this->Title) . "_{$this->CustomisableFormPanelID}";
		}
		parent::onBeforeWrite(); // TODO: Change the autogenerated stub

	}

	public function validate()
	{
		$result = parent::validate();
		$duplicates = CustomisableFormField::get()->filter(array("FieldTitle" => $this->FieldTitle, "CustomisableFormPanel.CustomisableFormID" => $this->CustomisableFormPanel()->CustomisableForm()->ID));
		if ($this->ID > 0){
			$duplicates = $duplicates->exclude(array("ID" => $this->ID));
		}
		if ($duplicates->count() > 0){
			$result->error('You already have another input in this form with the same title.');
		}
		return $result;
	}

}