<?php

namespace UncleCheese\BetterButtons\Extensions;

use Exception;
use SilverStripe\Core\Config\Config;
use SilverStripe\Core\Injector\Injector;
use SilverStripe\Forms\CompositeField;
use SilverStripe\Forms\FieldList;
use SilverStripe\ORM\DataExtension;
use SilverStripe\Versioned\Versioned;
use UncleCheese\BetterButtons\Buttons\BetterButton;
use UncleCheese\BetterButtons\FormFields\DropdownFormAction;

/**
 * An extension that offers features to DataObjects that allow them to set their own
 * actions and utilities for {@link GridFieldDetailForm}
 *
 * Default buttons are defined in _config.yml and can be overriden via the Config layer.
 * Due to the way Config merges arrays, set button class names to "false" to remove them from the list.
 *
 * @author  Uncle Cheese <unclecheese@leftandmain.com>
 * @package  silverstripe-gridfield-betterbuttons
 */
class BetterButtonDataObject extends DataExtension
{
    /**
     * Enable better buttons for this DataObject
     *
     * @config
     * @var bool
     */
    private static $better_buttons_enabled = true;

    /**
     * Enable versioned controls like 'Save & Publish' for DataObjects
     * with 'Versioned' extension.
     *
     * Disable this for records where you want a parent DataObject to control the
     * published/unpublished state of its children. ie. User Defined Forms 3.0+.
     *
     * @config
     * @var bool
     */
    private static $better_buttons_versioned_enabled = true;

    /**
     * Gets the default actions for all DataObjects. Can be overloaded in subclasses
     * <code>
     *  public function getBetterButtonsActions()
     *  {
     *      $actions = parent::getBetterButtonsActions();
     *      $actions->push(BetterButtonCustomAction::create('myaction','Do something to this record'));
     *
     *      return $actions;
     *  }
     * </code>
     *
     * @return FieldList
     */
    public function getBetterButtonsActions()
    {
        $buttons = $this->getDefaultButtonList("BetterButtonsActions");
        $actions = $this->createFieldList($buttons);

        $this->owner->extend('updateBetterButtonsActions', $actions);

        return $actions;
    }

    /**
     * Gets a FormAction or BetterButtonCustomAction by name, in utils or actions
     * @param  string $action  The name of the action to find
     * @return FormAction
     */
    public function findActionByName($action)
    {
        $actions = $this->owner->getBetterButtonsActions();
        $formAction = false;

        foreach ($actions as $f) {
            if ($formAction) {
                break;
            }

            if ($f instanceof CompositeField) {
                $formAction = $f->fieldByName($action);
            } elseif ($f->getName() === $action) {
                $formAction = $f;
            }
        }

        if (!$formAction) {
            $utils = $this->owner->getBetterButtonsUtils();
            $formAction = $utils->fieldByName($action);
        }

        return $formAction;
    }

    /**
     * Gets the default utils for all DataObjects. Can be overloaded in subclasses.
     * Utils are actions that appear in the top of the GridFieldDetailForm
     * <code>
     *  public function getBetterButtonsUtils()
     *  {
     *      $utils = parent::getBetterButtonsUtils();
     *      $utils->push(BetterButtonCustomAction::create('myaction','Do something to this record'));
     *
     *      return $utils;
     *  }
     * </code>
     *
     * @return FieldList
     */
    public function getBetterButtonsUtils()
    {
        $buttons = $this->getDefaultButtonList("BetterButtonsUtils");
        $utils = $this->createFieldList($buttons);

        $this->owner->extend('updateBetterButtonsUtils', $utils);

        return $utils;
    }

    /**
     * Gets an array of all the default buttons as defined in the config
     * @param  array $config
     * @return array
     */
    protected function getDefaultButtonList($config)
    {
        $new = ($this->owner->ID == 0);
        $list = $new
            ? Config::inst()->get($config, $this->checkVersioned() ? "versioned_create" : "create")
            : Config::inst()->get($config, $this->checkVersioned() ? "versioned_edit" : "edit");

        return $list ?: array ();
    }

    /**
     * Transforms a list of configured buttons into a usable FieldList
     * @param  array                            $buttons An array of class names
     * @return FieldList
     */
    protected function createFieldList($buttons)
    {
        $actions = FieldList::create();
        foreach ($buttons as $buttonType => $bool) {
            if (!$bool || !$buttonType) {
                continue;
            }

            if (substr($buttonType, 0, 6) == "Group_") {
                $group = $this->createButtonGroup(substr($buttonType, 6));
                if ($group->children->exists()) {
                    $actions->push($group);
                }
            } elseif ($b = $this->instantiateButton($buttonType)) {
                $actions->push($b);
            }
        }

        return $actions;
    }

    /**
     * Transforms a given button class name into an actual object.
     * @param  string                           $className The class of the button
     * @param  Form                             $form      The form that will contain the button
     * @param  GridFieldDetailForm_ItemRequest  $request   The request that points to the form
     * @param  boolean                          $button    If the button should display as an input tag or a button
     * @return BetterButton
     * @throws Exception If the requested button type does not exist
     */
    protected function instantiateButton($className)
    {
        try {
            return Injector::inst()->create($className);
        } catch (Exception $ex) {
            // Customize the default injector exception
            throw new Exception("The button type $className doesn't exist.");
        }
    }

    /**
     * Creates a button group {@link DropdownFormAction}
     * @param  string                           $groupName The name of the group
     * @return DropdownFormAction
     */
    protected function createButtonGroup($groupName)
    {
        $groupConfig = Config::inst()->get("BetterButtonsGroups", $groupName);
        $label = (isset($groupConfig['label'])) ? $groupConfig['label'] : $groupName;
        $buttons = (isset($groupConfig['buttons'])) ? $groupConfig['buttons'] : array ();
        $button = DropdownFormAction::create(_t('GridFieldBetterButtons.'.$groupName, $label));
        foreach ($buttons as $b => $bool) {
            if ($bool) {
                if ($child = $this->instantiateButton($b)) {
                    $button->push($child);
                }
            }
        }

        return $button;
    }

    /**
     * Determines if the record is using the {@link Versioned} extension
     * @return boolean
     */
    public function checkVersioned()
    {
        $isVersioned = false;

        foreach ($this->owner->getExtensionInstances() as $extension) {
            if ($extension instanceof Versioned) {
                $isVersioned = true;
                break;
            }
        }

        return $isVersioned
            && $this->owner->config()->better_buttons_versioned_enabled
            && count($this->owner->getVersionedStages()) > 1;
    }

    /**
     * Checks if a custom action is allowed to be called against a model.
     * Prevents security risk of calling arbitrary public methods on the DataObject.
     *
     * Looks at:
     * <code>
     *     private static $better_buttons_actions = array ()
     * </code>
     *
     * @param  string  $action The name of the action
     * @return boolean
     */
    public function isCustomActionAllowed($action)
    {
        $actions = $this->owner->config()->better_buttons_actions;
        if ($actions) {
            return in_array($action, $actions);
        }

        return false;
    }
}
